<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Models\Faq;
use App\Models\Contact;
use App\Models\FaqCategory;
use App\Models\Feedback;
use App\Models\ChatRoom;
use App\Models\FeedbackOption;
use App\Models\SampleQuestion;
use Carbon\Carbon;

class CommonController extends Controller
{
    public function contact(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'full_name' => 'required',
            'email'     => 'required|email',
        ], [
            'email.required'    => 'The email field is required.',
            'email.email'       => 'Please enter a valid email address.',
            'full_name.required'=> 'The password field is required.',
        ]);
        
        // If validation fails, return JSON response
        if ($validator->fails()) {
            return response()->json([
                'success'    => false,
                'message'   => 'Validation failed',
                'errors'    => $validator->errors()
            ], 200);
        }

        $created_at = Carbon::now();
        $user   = $request->user();

        $insert = Contact::insert([ 'user_id'   => $user->id,
                                    'full_name' => $request->full_name,
                                    'email'     => $request->email,
                                    'subject'   => $request->subject,
                                    'message'   => $request->message,
                                    'created_at'=> $created_at,
                                ]);

        if($insert)
        {
            return response()->json([
                'success'    => true,
                'message'   => 'Thank you for contacting us! We have received your message and will get back to you shortly.',
            ], 200);
        }
        else{
            return response()->json([
                'success'    => false,
                'message'   => 'Oops something went wrong',
            ], 200);
        }

    }

    public function feedbackoption(Request $request)
    {
        $list = FeedbackOption::where('status',1)->get(['id','name']);

        return response()->json([
            'success'   => true,
            'list'      => $list,
        ], 200);
    }


    public function feedback(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'rating'            => 'required|numeric|digits_between:1,5',
            'feedbackOptions'   => 'required|array',
        ]);
        
        // If validation fails, return JSON response
        if ($validator->fails()) {
            return response()->json([
                'success'    => false,
                'message'   => 'Validation failed',
                'errors'    => $validator->errors()
            ], 200);
        }

        $created_at = Carbon::now();
        $user   = $request->user();
        $oid    = json_encode($request->feedbackOptions);
        $insert = Feedback::insert([ 'user_id'   => $user->id,
                                    'rating'    => $request->rating,
                                    'oid'       => $oid,
                                    'message'   => $request->message,
                                    'created_at'=> $created_at,
                                ]);

        if($insert)
        {
            return response()->json([
                'success'    => true,
                'message'   => 'Thanks for sharing your thoughts! Your feedback helps us grow and serve you better.',
            ], 200);
        }
        else{
            return response()->json([
                'success'    => false,
                'message'   => 'Oops something went wrong',
            ], 200);
        }

    }

    public function sampleQuestion(Request $request)
    {
        $records = SampleQuestion::raw(function ($collection) {
    return $collection->aggregate([
        ['$sample' => ['size' => 5]],
        ['$project' => [
            '_id' => 1,
            'question' => 1
        ]]
    ]);
});
        return response()->json([
            'success'   => true,
            'message'   => 'Question List',
            'list'      => $records
        ], 200);
    }

    public function faq()
    {
        // Faq::where('status',1)->update(['status'=>'1']);
        $list = FaqCategory::select('id','name')
                                ->where('status','1')
                                ->with(['faqs' => function($query) {
                                    $query->select('id', 'question', 'answer','category_id')->where('status','1');
                                }])
                                ->get();

        return response()->json([
            'success'   => true,
            'message'   => 'Faq List',
            'list'      => $list
        ], 200);
    }

    public function togglePin(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id'            => 'required',
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success'    => false,
                'message'   => 'Validation failed',
                'errors'    => $validator->errors()
            ], 200);
        }
        
        $id         = $request->id;
        $chatRoom   = ChatRoom::findOrFail($id);

        $chatRoom->pin = abs($chatRoom->pin-1);
        $chatRoom->save();

        return response()->json([
            'success' => true,
            'message' => ($chatRoom->pin==1) ? 'Chat pinned' : 'Chat unpinned',
        ]);
    }
}
